import { useState, useEffect } from "react";
import { collection, addDoc, getDocs, query, where } from "firebase/firestore";
import db from "@/firebase";

const usersCollectionRef = collection(db, "newsletter subscribers");

const WaitlistPopup = () => {
  const [isOpen, setIsOpen] = useState(false);
  const [email, setEmail] = useState(""); // Track email input
  const [exists, setExists] = useState(false);
  const [subscribed, setSubscribed] = useState(false);
  const [errMsg, setErrMsg] = useState(false);

  // Show popup on page load
  useEffect(() => {
    setIsOpen(true);
  }, []);

  const handleSubmission = async (e) => {
    e.preventDefault();
    
    if (!email.trim()) return; // Prevent empty submissions

    try {
      // Query Firestore to check if email exists
      const emailQuery = query(usersCollectionRef, where("email", "==", email.toLowerCase()));
      const emailSnapshot = await getDocs(emailQuery);

      if (!emailSnapshot.empty) {
        setExists(true);
        setTimeout(() => setExists(false), 2000);
        return;
      }

      // Add email to Firestore
      await addDoc(usersCollectionRef, { email: email.toLowerCase() });
      setSubscribed(true);
      setIsOpen(false);
    } catch (error) {
      console.error("Error:", error);
      setErrMsg(true);
    } finally {
      // Reset states after 2 seconds
      setTimeout(() => {
        setSubscribed(false);
        setExists(false);
        setErrMsg(false);
        setEmail(""); // Reset email input
      }, 2000);
    }
  };

  return (
    <>
      {isOpen && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-[1200]">
          <div className="bg-white p-6 rounded-lg shadow-lg max-w-sm w-full relative mx-4">
            <div className="flex justify-between items-center">
              <h2 className="text-lg font-bold">Join the Waitlist</h2>
              <button
                onClick={() => setIsOpen(false)}
                className="text-gray-500 hover:text-gray-700"
              >
                ✕
              </button>
            </div>
            <p className="text-sm text-gray-600 mt-2">
              Be the first to get updates and exclusive offers!
            </p>
            <form className="mt-4" onSubmit={handleSubmission}>
              {subscribed && (
                <p className="text-green-400 text-sm font-medium absolute mt-[-18px]">
                  Subscribed successfully <span role="img" aria-label="thumbs-up">👍🏽</span>
                </p>
              )}
              {exists && (
                <p className="text-yellow-600 text-sm font-medium absolute mt-[-18px]">
                  Subscriber exists <span role="img" aria-label="thinking">🧐</span>
                </p>
              )}
              {errMsg && (
                <p className="text-yellow-600 text-sm font-medium absolute mt-[-18px]">
                  Oops... <span role="img" aria-label="sad">🙁</span>
                </p>
              )}
              <input
                type="email"
                placeholder="Enter your email"
                className="w-full p-2 border rounded-lg focus:outline-none focus:ring focus:ring-green-300"
                required
                value={email} // Controlled input
                onChange={(e) => setEmail(e.target.value)}
              />
              <button
                type="submit"
                className="mt-3 w-full bg-green-600 text-white p-2 rounded-lg hover:bg-green-700 transition"
              >
                Join Waitlist
              </button>
            </form>
          </div>
        </div>
      )}
    </>
  );
};

export default WaitlistPopup;
