import React, { useRef, useState } from "react";
import { useFormik } from "formik";
import * as yup from "yup";
import { collection, addDoc, query, where, getDocs } from "firebase/firestore";
import db from "../../firebase";

const Form = () => {
  const [loading, setLoading] = useState(false);
  const [exists, setExists] = useState(false);
  const [subscribed, setSubscribed] = useState(false);
  const [errMsg, setErrMsg] = useState(false);
  const usersCollectionRef = collection(db, "newsletter subscribers");
  const formRef = useRef(null);

  // Form validation schema
  const validationSchema = yup.object({
    email: yup.string().email("Enter a valid email").required("Email is required"),
  });

  // Formik setup
  const formik = useFormik({
    initialValues: { email: "" },
    validationSchema,
    onSubmit: async (values, { resetForm }) => {
      const email = values.email.toLowerCase();
      setLoading(true);

      try {
        // Query Firestore for existing email
        const emailQuery = query(usersCollectionRef, where("email", "==", email));
        const emailSnapshot = await getDocs(emailQuery);

        if (!emailSnapshot.empty) {
          setExists(true);
          setLoading(false);
          return;
        }

        // Add email to Firestore
        await addDoc(usersCollectionRef, { email });

        // Subscribe to newsletter API
        const response = await fetch(
          `https://newsletter-subscriber-api.onrender.com/api/memberAdd?email=${email}`
        );
        const data = await response.json();

        if (data.status === "subscribed") {
          setSubscribed(true);
        } else if (data.title === "Member Exists") {
          setExists(true);
        } else {
          setErrMsg(true);
        }
      } catch (error) {
        console.error("Error:", error);
        setErrMsg(true);
      } finally {
        setLoading(false);
        resetForm();
        setTimeout(() => {
          setSubscribed(false);
          setExists(false);
          setErrMsg(false);
        }, 2000);
      }
    },
  });

  return (
    <div className="w-full mt-6">
      <form className="w-full max-w-[31rem]" onSubmit={formik.handleSubmit} ref={formRef}>
        {formik.touched.email && formik.errors.email && (
          <p className="text-red-500 text-sm font-medium absolute mt-[-18px]">
            {formik.errors.email}
          </p>
        )}
        {subscribed && (
          <p className="text-green-400 text-sm font-medium absolute mt-[-18px]">
            Subscribed successfully <span role="img" aria-label="thumbs-up">👍🏽</span>
          </p>
        )}
        {exists && (
          <p className="text-yellow-600 text-sm font-medium absolute mt-[-18px]">
            Subscriber exists <span role="img" aria-label="thinking">🧐</span>
          </p>
        )}
        {errMsg && (
          <p className="text-yellow-600 text-sm font-medium absolute mt-[-18px]">
            Oops... <span role="img" aria-label="sad">🙁</span>
          </p>
        )}
        <input
          id="email"
          type="email"
          name="email"
          className="w-full h-12 max-w-[286px] text-black rounded-md px-4 text-[14px] outline-none mr-4"
          placeholder="youremail@example.com"
          onChange={formik.handleChange}
          onBlur={formik.handleBlur}
          value={formik.values.email}
        />
        <button
          type="submit"
          className="w-28 h-12 mt-3 lg:mt-0 text-white bg-[#55a654] prodigySans500 md:prodigySans600 rounded-md cursor-pointer hover:bg-[#55a654] disabled:bg-gray-400 disabled:cursor-not-allowed"
          disabled={loading}
        >
          {loading ? "Joining..." : "Subscribe"}
        </button>
        <p className="text-white mt-2 text-xs">
          <span>Join the family, be a part of our early users.</span>
          <br />
          Subscribe to our newsletter...
        </p>
      </form>
    </div>
  );
};

export default Form;
